;===============================================================================
; VGA-X Equates
;===============================================================================
ScreenStart   =       0
ScreenEnd     =       65535            ; LAST byte of video memory.
CRTC_Index    EQU     003D4h           ; CRTC Index Register
Status_Reg    EQU     003DAh           ; Input Status 1 Register
SC_Index      EQU     003C4h           ; Sequence Controller Register
GC_Index      EQU     003CEh           ; Graphics Controller Index
Misc_Index    EQU     003C2h           ; Miscelaneous register.
VideoSeg      EQU     0A000h           ; Segment Address of Video Segment
Map_Mask      EQU     00002h           ; Map Mask Register Index in SC
Start_High    EQU     0Ch              ; Start address high register
Start_Low     EQU     0Dh              ; Start address low register
ScreenWidth   EQU     320              ;
ScreenHeight  EQU     200              ;
BytesWidth    EQU     ScreenWidth/4    ;


;===============================================================================
; MACRO'S
;===============================================================================


;-------------------------------------------------------------------------------
;  SetSpeedBar (Color)
;    If 'SpeedBar' is defined, include code to display the speedbar so
;    you can see during te demo where the CPU time is going to
;-------------------------------------------------------------------------------
SetSpeedBar   MACRO  Color
              ifdef  SpeedBar
              MOV    DX,003C8H         ; Palette Address register
              MOV    AL,0              ; Change color 0
              OUT    DX,AL             ; Activate
              INC    DX                ; PEL Data register
              MOV    AL,Color          ; Color to write
              OUT    DX,AL             ; Activate Red
              OUT    DX,AL             ; Activate Green
              OUT    DX,AL             ; Activate Blue
              endif
              ENDM


;-------------------------------------------------------------------------------
;  VSync
;    Synchronize with Vertical Retrace.
;-------------------------------------------------------------------------------
VSync         MACRO
              LOCAL   Vs_EndSET,Vs_OkSet
              MOV     DX,003DAh        ; DX = Video Chip
Vs_EndSET:    IN      AL,DX            ; Get byte from Video Status port
              AND     AL,008h          ; Mask VSYNC
              JNZ     Vs_EndSET        ; If not VSYNC end try again
Vs_OkSET:     IN      AL,DX            ; Get byte from Video Status port
              AND     AL,008h          ; Mask VSYNC
              JZ      Vs_OkSET         ; If not VSYNC Begin try again
              ENDM


;-------------------------------------------------------------------------------
;  FlipPage  (Addres)
;    Flip the page to the offset defined by Address.
;-------------------------------------------------------------------------------
FlipPage      MACRO   Address
              LOCAL   WaitDE,WaitVS
; Wait for display enable to be active (status is active low), to be
; sure both halves of the start address will take in the same frame.
              MOV     BX,Address       ; BH = High part, BL = Low part
              MOV     CH,BL            ; CH = Low Part
              MOV     BL,Start_High    ; BX, value for Start address High
              MOV     CL,Start_Low     ; BX, value for Start address Low
              MOV     DX,Status_Reg
WaitDE:       IN      AL,DX
              TEST    AL,01h
              JNZ     WaitDE           ; Display enable is active
                                       ; low (0 = active)
; Set the start offset in display memory of the page to display.
              MOV     DX,CRTC_Index
              MOV     AX,BX
              OUT     DX,AX            ; Start address High
              MOV     AX,CX
              OUT     DX,AX            ; Start address low
; Now wait for vertical sync, so the other page will be invisible when
; we start drawing to it.
              MOV     DX,Status_Reg
              CLI
WaitVS:       IN      AL,DX
              TEST    AL,08h
              JZ      WaitVS           ; Vertical sync is active
                                       ; High (1 = active)
              STI
              ENDM


;-------------------------------------------------------------------------------
;  SplitScreen  (Line)
;    Split the screen at line (doesn't work on EGA)
;-------------------------------------------------------------------------------
SplitOff      EQU     03FFh            ; Turn SplitScreen Off.
SplitScreen   MACRO   Line
              VSync
              MOV     DX,CRTC_Index
              MOV     AX,Line          ; AX:= Value of scanline
              MOV     BH,AH            ; BH bit 0-1 := bit 8-9 of line compare value
              MOV     BL,BH            ; make a copy
              AND     BX,0201h         ; BH-Bit 1 = Bit 9 of line compare
                                       ; BL-Bit 0 = Bit 8 of line compare
              SHL     BX,4             ; BL-Bit 4 = Bit 8 of Line Compare
                                       ; BH-Bit 5 = Bit 9 of Line Compare
              SHL     BH,1             ; BH Bit 6 = Bit 9 of line compare
              ; Program registers
              MOV     AH,AL            ; AH := Low (Lijn)
              MOV     AL,018h          ; Line Compare Register
              OUT     DX,AX
              MOV     AL,7             ; Overflow register
              OUT     DX,AX
              INC     DX
              IN      AL,DX            ; AL:=Current value of overflow register
              DEC     DX

              MOV     AH,AL
              AND     AH,0EFh          ; Bit 4 := 0
              OR      AH,BL            ; Bit 4 := Bit 8 of line compare
              MOV     AL,7             ; Overflow register
              OUT     DX,AX

              MOV     AL,9             ; AL:=nr of Max Scan Line Register
              OUT     DX,AL
              INC     DX
              IN      AL,DX            ; AL := Curr. Value of max scan line register
              DEC     DX
              MOV     AH,AL
              AND     AH,0BFh          ; Bit 6:=0
              OR      AH,BH            ; Bit 6:= Bit 9 of line compare
              MOV     AL,9             ; AL := nr of max scan-line register
              OUT     DX,AX            ; change max scan line register
              ENDM


;------------------------------------------------------------------------------
;  SetPalette  (Palette, StartColor, Size)
;    Set the palette from color <Startcolor>, <Size> colors, at Offset
;    <PalOffset>
;------------------------------------------------------------------------------
SetPalette    MACRO  Palette, StartColor, Size
              MOV    DX,003C8H         ; Palette Address register
              MOV    AL,StartColor     ; Change color 0
              OUT    DX,AL             ; Activate
              INC    DX                ; PEL Data register
              MOV    SI,Offset Palette ; DS:SI -> Palette data
              MOV    CX,3*(Size)       ; # of bytes to set
              CLD                      ; Forwards on string instructions
              REP    OUTSB             ; Set all colours.
              ENDM


;------------------------------------------------------------------------------
;  LoadToScreen  (SourceOffs, DestOffs, Size)
;    Load <Size> consecutive bytes from DS:SourceOffs, to the DestOffs in
;    Video memory (unchained mode).
;------------------------------------------------------------------------------
LoadToScreen  MACRO   SourceOffs, DestOffs, Size
              LOCAL   NextPixel
              MOV     DX,GC_Index      ; \  Enable writes from all bits to
              MOV     AX,0FF08h        ;  > video memory.
              OUT     DX,AX            ; /
              MOV     DX,SC_Index      ; \
              MOV     AL,Map_Mask      ;  \ Activate Map Mask register.
              OUT     DX,AL            ;  /
              INC     DX               ; /
              MOV     AX,VideoSeg
              MOV     ES,AX
              CLD                      ; Forwards on string instructions.
              MOV     AL,011h          ; Mask for MapMask we'll be using.
              MOV     CX,Size
              MOV     SI,SourceOffs
              MOV     DI,DestOffs
NextPixel:    OUT     DX,AL
              ROL     AL,1
              MOVSB
              ADC     DI,-1            ; Backup byte, unless new byte needed
              LOOP    NextPixel
              ENDM


;------------------------------------------------------------------------------
; Set Tweaked Mode (320*200*256 unchained)
;------------------------------------------------------------------------------
SetTweaked    PROC    NEAR
              MOV     AX,13h          ; Let the BIOS do most of the work
              INT     10h             ; by installing mode 13h first

              MOV     DX,SC_Index     ; Sequence Controller Register
              MOV     AL,4
              OUT     DX,AL
              INC     DX
              IN      AL,DX
              AND     AL,0F7h
              OR      AL,04h
              OUT     DX,AL

              MOV     DX,GC_Index     ; Graphics Controller Index
              MOV     AL,5
              OUT     DX,AL
              INC     DX
              IN      AL,DX
              AND     AL,0EFh
              OUT     DX,AL

              MOV     DX,GC_Index     ; Graphics Controller Index
              MOV     AL,6
              OUT     DX,AL
              INC     DX
              IN      AL,DX
              AND     AL,0FDh
              OUT     DX,AL

              MOV     DX,SC_Index     ; Sequence Controller Register
              MOV     AX,0F02h
              OUT     DX,AX           ; enable writes to all planes
              MOV     AX,VideoSeg     ; \ ES=Video Segment
              MOV     ES,AX           ; /
              XOR     DI,DI           ; ES:DI -> Begin of Screen Data
              XOR     AX,AX           ; Data to write
              MOV     CX,8000h        ; # Words in display memory
              REP     STOSW           ; Clear entire Video Segment

              MOV     DX,CRTC_Index   ; CRTC Index Register
              MOV     AL,14h
              OUT     DX,AL
              INC     DX
              IN      AL,DX
              AND     AL,0BFh
              OUT     DX,AL

              MOV     DX,CRTC_Index   ; CRTC Index Register
              MOV     AL,17h
              OUT     DX,AL
              INC     DX
              IN      AL,DX
              OR      AL,40h
              OUT     DX,AL

              RET
SetTweaked    ENDP


;------------------------------------------------------------------------------
; Set Mode X. (320*240*256)
;------------------------------------------------------------------------------
NumParms      EQU     10
CRTC_Parms    DW      00D06h           ; Vertical Total
              DW      03E07h           ; Overflow (bit 8 of vertical counts
              DW      04109h           ; Cell Height (2 to double scan)
              DW      0EA10h           ; VSync Start
              DW      0AC11h           ; VSync End and protect cr0-7
              DW      0DF12h           ; Vertical Displayed
              DW      00014h           ; turn off DWORD mode
              DW      0E715h           ; v blank start
              DW      00616h           ; v blank end
              DW      0E317h           ; turn on byte mode

SetModeX      PROC    NEAR

              MOV     DX,MISC_Index    ; Miscellaneous Output register
              MOV     AX,0E3h          ; \ Select 28 MHz dot clock & 60 MHz
              OUT     DX,AX            ; / scanning rate

              MOV     DX,SC_Index      ; Sequence Controller Register
              MOV     AX,0300h
              OUT     DX,AX            ; undo reset (restart sequencer

              MOV     DX,CRTC_Index    ; CRTC Controller register
              MOV     AL,11h           ; \ VSync end reg contains register
              OUT     DX,AL            ; / write protect bit
              INC     DX               ; CRTC Data register
              IN      AL,DX            ; Get curr VSync End register setting
              AND     AL,07Fh          ; \ Remove write protect on various
              OUT     DX,AL            ; / CRTC Registers
              DEC     DX               ; CRTC Controller index
              CLD
              MOV     SI,Offset CRTC_Parms; DS:SI -> CRTParms
              MOV     CX,NumParms      ; #of Params to change
SetCRT:       LODSW                    ; Get next index/data pair
              OUT     DX,AX            ; write it
              LOOP    SetCRT           ; loop for all params

              RET
SetModeX      ENDP


;------------------------------------------------------------------------------
; Set RasterHeight
;------------------------------------------------------------------------------
; Set the raster size of the VGA.
; 0 is 1 scanline, 1 is double scanning....
;------------------------------------------------------------------------------
SetRaster     MACRO   Size
              MOV     DX,CRTC_Index
              MOV     AL,09h
              OUT     DX,AL
              INC     DX
              IN      AL,DX
              MOV     AL,Size
              OR      AL,BL
              OUT     DX,AX
              ENDM








